/**************************************************************************
 * Filename: Interrupt.c
 *
 * 
 * Copyright (c) 2011, Freescale Semiconductor, Inc. All rights reserved.
 *
 ***************************************************************************
 *
 * THIS SOFTWARE IS PROVIDED BY FREESCALE "AS IS" AND ANY EXPRESSED OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL FREESCALE OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 *
 ***************************************************************************/




/******************************************************************************
*******************************************************************************
* Includes
*******************************************************************************
******************************************************************************/


#include "PortConfig.h"
#include "Interrupt.h"
#include "EmbeddedTypes.h"
#if defined(IAR) 
#include "intrinsics.h"	 
#endif
#if defined (CW)
#include "UartUtil.h"
#endif 

extern int (*pfDisableProccessorIrqs)(void);
extern void (*pfEnableProccessorIrqs)(int);

#if defined (CW)
void IntDisableAll(uint32_t var);
void IntRestoreAll(uint32_t var1);
#endif

/***********************************************************************/
/*
 * Configures the ARM system control register for STOP (deep sleep) mode
 * and then executes the WFI instruction to enter the mode.
 *
 * Parameters:
 * none
 *
 * Note: Might want to change this later to allow for passing in a parameter
 *       to optionally set the sleep on exit bit.
 */

void stop (void)
{
	/* Set the SLEEPDEEP bit to enable deep sleep mode (STOP) */
	SCB_SCR |= SCB_SCR_SLEEPDEEP_MASK;	


	/* WFI instruction will start entry into STOP mode */
#ifndef KEIL
	asm("WFI");
#else
	__wfi();
#endif
}
/***********************************************************************/
/*
 * Configures the ARM system control register for WAIT (sleep) mode
 * and then executes the WFI instruction to enter the mode.
 *
 * Parameters:
 * none
 *
 * Note: Might want to change this later to allow for passing in a parameter
 *       to optionally set the sleep on exit bit.
 */

void wait (void)
{
	/* Clear the SLEEPDEEP bit to make sure we go into WAIT (sleep) mode instead
	 * of deep sleep.
	 */
	SCB_SCR &= ~SCB_SCR_SLEEPDEEP_MASK;	

	/* WFI instruction will start entry into WAIT mode */
#ifndef KEIL
	asm("WFI");
#else
	__wfi();
#endif
}
/***********************************************************************/
/*
 * Change the value of the vector table offset register to the specified value.
 *
 * Parameters:
 * vtor     new value to write to the VTOR
 */

void write_vtor (int vtor)
{
        /* Write the VTOR with the new value */
        SCB_VTOR = vtor;	
}


/*---------------------------------------------------------------------------
* NAME: NVIC_EnableIRQ
* DESCRIPTION: Enables the specified IRQ (refer to Kinetis manual) 
* PARAMETERS: irq - IRQ number
* RETURN: -
* NOTES:  Interrupts will also need to be enabled in the ARM core. This can be 
* done using the EnableInterrupts macro.
*---------------------------------------------------------------------------*/


void NVIC_EnableIRQ (int irq)
{
    /* Make sure that the IRQ is an allowable number. Right now up to 48 is 
     * used.
     */
    if (irq > ARM_MAX_IRQ_VALUE)
		#if defined(IAR)
		printf ("\nERR! Invalid IRQ value passed to enable irq function!\n");	
                #elif  defined (CW)
                UartUtil_Print("\nERR! Invalid IRQ value passed to enable irq function!\n\r", gAllowToBlock_d);
                #endif
    	
    else
    {
      /* Determine which of the NVICISERs corresponds to the irq */
      NVIC_ICPR |= 1 << (irq%32);
      NVIC_ISER |= 1 << (irq%32);
    }             
}


/*---------------------------------------------------------------------------
* NAME: NVIC_DisableIRQ
* DESCRIPTION: Disables the specified IRQ (refer to Kinetis manual) 
* PARAMETERS: irq - IRQ number to be disabled
* RETURN: -
* NOTES:  If you want to disable all interrupts, then use the 
* DisableInterrupts macro instead.
*---------------------------------------------------------------------------*/

void NVIC_Disable_Irq (int irq)
{
    /* Make sure that the IRQ is an allowable number. Right now up to 91 is 
     * used.
     */
	
    if (irq > ARM_MAX_IRQ_VALUE)
		#if defined(IAR)
                printf ("\nERR! Invalid IRQ value passed to disable irq function!\n");	
		#elif  defined (CW)	
                UartUtil_Print("\nERR! Invalid IRQ value passed to disable irq function!\n\r",gAllowToBlock_d);
                #endif
    	
    else
      /* Determine which of the NVICICERs corresponds to the irq */
      NVIC_ICER = 1 << (irq%32);            
}


/*---------------------------------------------------------------------------
* NAME: NVIC_SetPriority
* DESCRIPTION: Sets the specified IRQ priority (refer to Kinetis manual) 
* PARAMETERS: irq - IRQ number
* 			  	
* RETURN: -
* NOTES: none  
*---------------------------------------------------------------------------*/

void NVIC_SetPriority (int irq, int prio)
{
    
    /* Make sure that the IRQ is an allowable number. Right now up to 91 is 
     * used.
     */
    if (irq > ARM_MAX_IRQ_VALUE)
		#if defined(IAR)
		printf ("\nERR! Invalid IRQ value passed to priority irq function!\n");		
		#elif defined (CW)
                UartUtil_Print("\nERR! Invalid IRQ value passed to priority irq function!\n\r",gAllowToBlock_d);
                #endif
                
    if (prio > ARM_MAX_PRIO_VALUE)
		#if defined(IAR)
		printf ("\nERR! Invalid priority value passed to priority irq function!\n");	
                #elif  defined (CW)
                UartUtil_Print("\nERR! Invalid priority value passed to priority irq function!\n\r",gAllowToBlock_d);
                #endif
           
}

#if defined(IAR) 
/***********************************************************************************/
void IntRestoreAll(uint32_t var)
{
    asm(" MSR PRIMASK, %0" : : "r" (var));  
    asm(" CPSIE i");
}

/***********************************************************************************/
uint32_t IntDisableAll(void)
{
	uint32_t var;
	
	asm("MRS  %0, PRIMASK" : "=r" (var));                                   //Read PRIMASK value and write it to R0
	asm(" CPSID i");
	
	return var;
}
/***********************************************************************************/
void InterruptInit(void)
{
  pfDisableProccessorIrqs = (int(*)(void))IntDisableAll;
  pfEnableProccessorIrqs = (void(*)(int))IntRestoreAll;
}

/***********************************************************************************/
#endif /*if defined (IAR)*/

#if defined(CW) 
//TODO!! Need to develop these routines in assembly for CodeWarrior
/***********************************************************************************/
void IntRestoreAll(uint32_t var)
{ 
    asm(" MSR PRIMASK, %0" : : "r" (var));  
    asm(" CPSIE i");
}

/***********************************************************************************/
uint32_t IntDisableAll(void)
{
	uint32_t var;
	
	asm("MRS  %0, PRIMASK" : "=r" (var));                                   //Read PRIMASK value and write it to R0
	asm(" CPSID i");
	
	return var;
}
/***********************************************************************************/
void InterruptInit(void)
{

//TODO!!	
//TODO Check this function pointer declaration
//  pfDisableProccessorIrqs = (void(*)(uint32_t))IntDisableAll;
//  pfEnableProccessorIrqs = (void(*)(uint32_t))IntRestoreAll;
}

/***********************************************************************************/
#endif /*if defined (CW)*/

